/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * (c) Jonathan Hudson <jh+mwptools@daria.co.uk>
 */

/*
  Serves black tiles for the map background. Useful for demo / videos, as a
  artefacts like WPs, range circles etc are very obvious.

  Requires an entry in ~/.config/mwp/sources.json like:

  {
    "id": "Black",
    "name": "Black Tiles",
    "license": "(c) jh ",
    "license_uri": "http://daria.co.uk/",
    "min_zoom": 0,
    "max_zoom": 20,
    "tile_size": 256,
    "projection": "MERCATOR",
    "spawn" : "bproxy",
  }

 Then put bproxy on the PATH (e.g. /use/local/bin). It is not built or
 installed automatically

 Note there is no need to define a URI (or port)
  */


public class BProxy : Soup.Server
{
    private uint8 []black_png =  {
        0x89, 0x50, 0x4e, 0x47, 0x0d, 0x0a, 0x1a, 0x0a, 0x00, 0x00, 0x00, 0x0d,
        0x49, 0x48, 0x44, 0x52, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x08, 0x02, 0x00, 0x00, 0x00, 0xd3, 0x10, 0x3f, 0x31, 0x00, 0x00, 0x01,
        0x83, 0x69, 0x43, 0x43, 0x50, 0x49, 0x43, 0x43, 0x20, 0x70, 0x72, 0x6f,
        0x66, 0x69, 0x6c, 0x65, 0x00, 0x00, 0x28, 0x91, 0x7d, 0x91, 0x3d, 0x48,
        0xc3, 0x40, 0x1c, 0xc5, 0x5f, 0x53, 0xa5, 0x22, 0x95, 0x0e, 0x76, 0x10,
        0x71, 0x88, 0x50, 0x9d, 0x2c, 0x88, 0x8a, 0x38, 0x6a, 0x15, 0x8a, 0x50,
        0x21, 0xd4, 0x0a, 0xad, 0x3a, 0x98, 0x5c, 0xfa, 0x05, 0x4d, 0x1a, 0x92,
        0x14, 0x17, 0x47, 0xc1, 0xb5, 0xe0, 0xe0, 0xc7, 0x62, 0xd5, 0xc1, 0xc5,
        0x59, 0x57, 0x07, 0x57, 0x41, 0x10, 0xfc, 0x00, 0x71, 0x72, 0x74, 0x52,
        0x74, 0x91, 0x12, 0xff, 0x97, 0x14, 0x5a, 0xc4, 0x78, 0x70, 0xdc, 0x8f,
        0x77, 0xf7, 0x1e, 0x77, 0xef, 0x00, 0xa1, 0x51, 0x61, 0x9a, 0xd5, 0x35,
        0x0e, 0x68, 0xba, 0x6d, 0xa6, 0x93, 0x09, 0x31, 0x9b, 0x5b, 0x15, 0x43,
        0xaf, 0x10, 0x10, 0x46, 0x04, 0xc3, 0x08, 0xc8, 0xcc, 0x32, 0xe6, 0x24,
        0x29, 0x05, 0xdf, 0xf1, 0x75, 0x8f, 0x00, 0x5f, 0xef, 0xe2, 0x3c, 0xcb,
        0xff, 0xdc, 0x9f, 0xa3, 0x4f, 0xcd, 0x5b, 0x0c, 0x08, 0x88, 0xc4, 0xb3,
        0xcc, 0x30, 0x6d, 0xe2, 0x0d, 0xe2, 0xe9, 0x4d, 0xdb, 0xe0, 0xbc, 0x4f,
        0x1c, 0x65, 0x25, 0x59, 0x25, 0x3e, 0x27, 0x1e, 0x33, 0xe9, 0x82, 0xc4,
        0x8f, 0x5c, 0x57, 0x3c, 0x7e, 0xe3, 0x5c, 0x74, 0x59, 0xe0, 0x99, 0x51,
        0x33, 0x93, 0x9e, 0x27, 0x8e, 0x12, 0x8b, 0xc5, 0x0e, 0x56, 0x3a, 0x98,
        0x95, 0x4c, 0x8d, 0x78, 0x8a, 0x38, 0xa6, 0x6a, 0x3a, 0xe5, 0x0b, 0x59,
        0x8f, 0x55, 0xce, 0x5b, 0x9c, 0xb5, 0x4a, 0x8d, 0xb5, 0xee, 0xc9, 0x5f,
        0x18, 0xce, 0xeb, 0x2b, 0xcb, 0x5c, 0xa7, 0x39, 0x84, 0x24, 0x16, 0xb1,
        0x04, 0x09, 0x22, 0x14, 0xd4, 0x50, 0x46, 0x05, 0x36, 0xe2, 0xb4, 0xea,
        0xa4, 0x58, 0x48, 0xd3, 0x7e, 0xc2, 0xc7, 0x3f, 0xe8, 0xfa, 0x25, 0x72,
        0x29, 0xe4, 0x2a, 0x83, 0x91, 0x63, 0x01, 0x55, 0x68, 0x90, 0x5d, 0x3f,
        0xf8, 0x1f, 0xfc, 0xee, 0xd6, 0x2a, 0x4c, 0x4e, 0x78, 0x49, 0xe1, 0x04,
        0xd0, 0xfd, 0xe2, 0x38, 0x1f, 0x23, 0x40, 0x68, 0x17, 0x68, 0xd6, 0x1d,
        0xe7, 0xfb, 0xd8, 0x71, 0x9a, 0x27, 0x40, 0xf0, 0x19, 0xb8, 0xd2, 0xdb,
        0xfe, 0x6a, 0x03, 0x98, 0xf9, 0x24, 0xbd, 0xde, 0xd6, 0x62, 0x47, 0x40,
        0x64, 0x1b, 0xb8, 0xb8, 0x6e, 0x6b, 0xca, 0x1e, 0x70, 0xb9, 0x03, 0x0c,
        0x3c, 0x19, 0xb2, 0x29, 0xbb, 0x52, 0x90, 0xa6, 0x50, 0x28, 0x00, 0xef,
        0x67, 0xf4, 0x4d, 0x39, 0xa0, 0xff, 0x16, 0xe8, 0x5d, 0xf3, 0x7a, 0x6b,
        0xed, 0xe3, 0xf4, 0x01, 0xc8, 0x50, 0x57, 0xa9, 0x1b, 0xe0, 0xe0, 0x10,
        0x18, 0x2d, 0x52, 0xf6, 0xba, 0xcf, 0xbb, 0x7b, 0x3a, 0x7b, 0xfb, 0xf7,
        0x4c, 0xab, 0xbf, 0x1f, 0x07, 0xb8, 0x72, 0x7c, 0xb8, 0x7c, 0x63, 0x40,
        0x00, 0x00, 0x00, 0x09, 0x70, 0x48, 0x59, 0x73, 0x00, 0x00, 0x2e, 0x23,
        0x00, 0x00, 0x2e, 0x23, 0x01, 0x78, 0xa5, 0x3f, 0x76, 0x00, 0x00, 0x00,
        0x07, 0x74, 0x49, 0x4d, 0x45, 0x07, 0xe4, 0x02, 0x0c, 0x0f, 0x22, 0x33,
        0x62, 0x29, 0x7f, 0xe7, 0x00, 0x00, 0x00, 0x19, 0x74, 0x45, 0x58, 0x74,
        0x43, 0x6f, 0x6d, 0x6d, 0x65, 0x6e, 0x74, 0x00, 0x43, 0x72, 0x65, 0x61,
        0x74, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x47, 0x49, 0x4d,
        0x50, 0x57, 0x81, 0x0e, 0x17, 0x00, 0x00, 0x00, 0xd5, 0x49, 0x44, 0x41,
        0x54, 0x78, 0xda, 0xed, 0xc1, 0x31, 0x01, 0x00, 0x00, 0x00, 0xc2, 0xa0,
        0xf5, 0x4f, 0xed, 0x65, 0x0b, 0xa0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x80, 0x1b, 0x01, 0x2d, 0x00, 0x01, 0x9e, 0xcb,
        0xc2, 0xed, 0x00, 0x00, 0x00, 0x00, 0x49, 0x45, 0x4e, 0x44, 0xae, 0x42,
        0x60, 0x82
    };
	private uint8 []base_png;

    public BProxy() {
		var fn = Environment.get_variable("MWP_BLACK_TILE");
		ssize_t nb = 0;
		if(fn != null) {
			Posix.Stat st;
			if(Posix.stat(fn, out st) == 0) {
				base_png = new uint8[st.st_size];
				int fd = Posix.open (fn, Posix.O_RDONLY);
				nb = Posix.read(fd, base_png, (int)st.st_size);
				Posix.close(fd);
			}
		}
		if (nb == 0 || base_png == null)
			base_png = black_png;

        this.add_handler (null, default_handler);
    }

    private void default_handler (Soup.Server server, Soup.ServerMessage msg, string path,
                          GLib.HashTable? query) {
        msg.set_response ("image/png", Soup.MemoryUse.COPY, base_png);
        msg.set_status(200, null);
    }

    public static int main (string []args) {
        var loop = new MainLoop();
        var b = new BProxy();
        try {
            b.listen_local(0, 0);
        } catch (Error e) {
            stdout.printf ("Error: %s\n", e.message);
        }
        var port = b.get_uris().nth_data(0).get_port ();
        print ("Port: %u\n", port);
        loop.run();
        return 0;
    }
}
